/*
 *  records.js
 *
 *  Copyright: (c) 2018 FileMaker, Inc. All rights reserved.
 *
 *  FILEMAKER CONFIDENTIAL
 *  This file includes unpublished proprietary source code of FileMaker,
 *  Inc. The copyright notice above does not evidence any actual or
 *  intended publication of such source code.
 */
'use strict';
var errorHandler = require('../routes/error');
var util = require('../routes/util');

var recordId;

/**
 * @api {GET} /databases/:solution/layout/:layout[?offset=:startRecordOffset&range=:rangeOfRecords&sort=:[{field_sort_config},..] Get Records
 * @apiDescription Get all or subset of records in specified solution layout.
 * All records will be included in result set if not query parameters are given. <br>
 * Optional query parameters "offset" and "range" can be used to specify starting record offset and range in result set.<br>
 * Optional query parameter "sort" uses JSON literal to specify the order of result set. <br>
 * Note: <strong>"offset" will be ignored if "sort" is in used. </strong> <br>
 * All values in query parameter must be percent-encoded as URI component.
 * @apiName getAllRecord
 * @apiGroup Record
 *
 * @apiParamExample {json} Query with offset and range
 *	?offset=1&range=50
 *
 * @apiParamExample {json} Query with range and sort
 *	?&range=50&sort=[{ "fieldName": "Work State", "precedence": 1, "sortOrder": "ascend" }, { "fieldName": "Work City", "precedence": 2, "sortOrder": "descend" }]
 *
 * @apiUse AccessTokenHeader
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} data JSON Array of row objects
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "records": "[...]"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases/:solution/layout/:layout?_offset=:startRecordOffset&_limit=:rangeOfRecords&sort=:sortConf&script=:postScriptName&script.param=:optionalPostScriptParam
 */
module.exports.getRecords = function(req, res, next) {
     var ipStr = req.get('X-Forwarded-For');  //bug 236856, get X-Forwarded-For first IP as public client IP
     var ips = [];
     if (!ipStr) {
         ipStr = "";
     } else {
         ips = ipStr.split(',');
         ipStr = ips[0];
     }

	 var params = {
		 'solution': req.swagger.params.database.value,
		 'layout': req.swagger.params.layout.value,
		 'requestMethod': req.method,
		 'requestPath': req.originalUrl,
		 'requestIp': ipStr,
		 'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		 'version': req.swagger.params.version.value
		};

	var result = util.setAuthentication(req, res, params, false);	
	if(result){
		for (var propName in req.query) {
			// enforce that only supported query parameters are allowed
			if( (['_offset','_limit','_sort','portal','script.param','script.prerequest', 'script.prerequest.param','script.presort','script.presort.param','script','layout.response','_zapier', 'dateformats'].indexOf(propName) === -1) &&
					!propName.toLowerCase().startsWith("_offset.") &&
					!propName.toLowerCase().startsWith("_limit.") )
				return errorHandler.handleError('BadRequest', req, res, "Unknown parameter(s): "+propName,"960");

			if ( (propName.toLowerCase().startsWith("_offset") || propName.toLowerCase().startsWith("_limit")) && (isNaN(req.query[propName]) || req.query[propName]<=0) ) {
				return errorHandler.handleError('BadRequest', req, res, 'The '+ propName +' must be an integer with value greater than 0.',"960")
			}
			if ( (propName.toLowerCase().startsWith("dateformats")) && (isNaN(req.query[propName]) || req.query[propName]<0 || req.query[propName]>2) ) {
				return errorHandler.handleError('BadRequest', req, res, 'The '+ propName +' must be an integer with value 0, 1 or 2.',"960")
			}
			var val = req.query[propName];
			if (val instanceof Array) {
				// more than one value, take the last one
				return errorHandler.handleError('BadRequest', req, res, 'Request contains multiple '+ propName +' parameters',"960")
			} else {
				if (propName.charAt(0) == '_') {
					params[propName.toLowerCase().substring(1)] = req.query[propName];
				} else {
					params[propName.toLowerCase()] = req.query[propName];
				}
			}
		}

		if (req.query.hasOwnProperty('script.param') && req.query['script.param'] && !req.query.hasOwnProperty('script')) {
			return errorHandler.handleError('BadRequest', req, res, "script.param without script","960");
		} else if (req.query.hasOwnProperty('script.prerequest.param') && req.query['script.prerequest.param'] && !req.query.hasOwnProperty('script.prerequest')) {
			return errorHandler.handleError('BadRequest', req, res, "script.prerequest.param without script.prerequest","960");
		} else if (req.query.hasOwnProperty('script.presort.param') && req.query['script.presort.param'] && !req.query.hasOwnProperty('script.presort')) {
			return errorHandler.handleError('BadRequest', req, res, "script.presort.param without script.presort","960");
		}

		try{
			util.thrift_client.getRecords(params,
				function(thrifError, thrifResult){
					return util.handleThrifReturn(thrifError, thrifResult, req, res);
			});
		} catch (err){
			util.thriftExceptionHandler(err, req, res);
		}
	}
};

/**
 * @api {POST} /databases/:solution/layout/:layout  Create Record
 * @apiDescription Create a new record for specified solution layout.
 * @apiName createRecord
 * @apiGroup Record
 *
 * @apiUse ContentTypeHeader
 * @apiUse AccessTokenHeader
 *
 * @apiParam {json} data	A stringify JSON object contains field-value pairs in the target layout.
 * A default filed-value pair will be applied if it is mising in data.
 * If "{}" is provided as data value, an empty record with default field-values will be created.
 *
 * @apiParamExample {json} Request-Example:
 * {"data": "{
 *       \"field_1\": \"value_1\",
 *       \"field_2\": \"value_2\"
 *       \"repetitionField(1)\" : \"fieldValue\"		//set the 1st fieldvalue in a repitition field
 *       \"Orders::OrderDate.0\":\"12/22/2015\",		//create a related "Orders" record with specify OrderDate, (use 0 for recordId to create)
 *     }"
 *	}
 *
 * @apiSuccess {String} errorCode "0"
 * @apiSuccess {String} recordId Record ID of newly created record
 *
 * @apiSuccessExample Success-Response:
 *     HTTP/1.1 200 OK
 *     {
 *       "result": "OK",
 *       "recordId": "newRecordId"
 *     }
 *
 * @apiUse BadRequestError
 * @apiUse UnauthorizedError
 * @apiUse FMServiceError
 *
 *
 * @apiSampleRequest /databases/:solution/layout/:layout
 */
module.exports.create = function(req, res, next) {
	if(!req.body['fieldData']){
		return errorHandler.handleError("BadRequest", req, res, "fieldData is missing.","10");
	}
	var data;
	if (req.body.hasOwnProperty('fieldData')) {
		data = (typeof req.body['fieldData'] === "object") ? JSON.stringify(req.body['fieldData']) : req.body['fieldData'].toString();
		if(data.trim().length===0){
			return errorHandler.handleError("BadRequest", req, res, "fieldData is empty.","10");
		}
		if (typeof JSON.parse(data) !== "object") {
			return errorHandler.handleError("BadRequest", req, res, "fieldData is not an object","960");
		}
		delete req.body['fieldData'];
	}

  	var ipStr = req.get('X-Forwarded-For');  //bug 236856, get X-Forwarded-For first IP as public client IP
    var ips = [];
    if (!ipStr) {
        ipStr = "";
    } else {
        ips = ipStr.split(',');
        ipStr = ips[0];
	}

	var params = {
		'fieldData': data,
		'solution': req.swagger.params.database.value,
		'layout': req.swagger.params.layout.value,
		'requestMethod': req.method,
		'requestPath': req.originalUrl,
		'requestIp': ipStr,
		'inputBandwidth': (req.body) ? JSON.stringify(req.body).length.toString() : "0",
		'version': req.swagger.params.version.value
    };
	var result = util.setAuthentication(req, res, params, false);	
	if(result){
		if (req.body.hasOwnProperty('portalData')) {
			if (req.body['portalData']) {
				if (req.body['portalData'] !== Object(req.body['portalData'])) {
					return errorHandler.handleError("BadRequest", req, res, "portalData is not an object","960");
				}
				for (var portalName in req.body['portalData']) {
					var portalRowArray = req.body['portalData'][portalName];
					if (!Array.isArray(portalRowArray)) {
						return errorHandler.handleError("BadRequest", req, res, "'"+ portalName +"' in portalData is not an array","960");	
					}
				}
				params.portalData = (typeof req.body['portalData'] === "object") ? JSON.stringify(req.body['portalData']) : req.body['portalData'].toString();
				if (params.portalData.trim().length===0) {
					return errorHandler.handleError("BadRequest", req, res, "portalData is empty.","10");
				}
				delete req.body['portalData'];
			} else {
				return errorHandler.handleError("BadRequest", req, res, "portalData is empty.","10");
			}
		}
		if (req.body.hasOwnProperty('script') && req.body['script']) {
			if (typeof req.body['script'] !== "string" && !(req.body['script'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script value is not a string" ,"960");
			}
			params['script'] = req.body['script'];
			delete req.body['script'];
			if (req.body.hasOwnProperty('script.param') && req.body['script.param']) {
				if (typeof req.body['script.param'] !== "string" && !(req.body['script.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.param value is not a string" ,"960");
				}
				params['script.param'] = req.body['script.param'];
				delete req.body['script.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.param without script" ,"960");		
		}
		if (req.body.hasOwnProperty('script.prerequest') && req.body['script.prerequest']) {
			if (typeof req.body['script.prerequest'] !== "string" && !(req.body['script.prerequest'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script.prerequest value is not a string" ,"960");
			}
			params['script.prerequest'] = req.body['script.prerequest'];
			delete req.body['script.prerequest'];
			if (req.body['script.prerequest.param']) {
				if (typeof req.body['script.prerequest.param'] !== "string" && !(req.body['script.prerequest.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.prerequest.param value is not a string" ,"960");
				}
				params['script.prerequest.param'] = req.body['script.prerequest.param'];
				delete req.body['script.prerequest.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.prerequest.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.prerequest.param without script.prerequest" ,"960");		
		}
		if (req.body.hasOwnProperty('script.presort') && req.body['script.presort']) {
			if (typeof req.body['script.presort'] !== "string" && !(req.body['script.presort'] instanceof String)) {
				return errorHandler.handleError("BadRequest", req, res, "script.presort value is not a string" ,"960");
			}
			params['script.presort'] = req.body['script.presort'];
			delete req.body['script.presort'];
			if (req.body['script.presort.param']) {
				if (typeof req.body['script.presort.param'] !== "string" && !(req.body['script.presort.param'] instanceof String)) {
					return errorHandler.handleError("BadRequest", req, res, "script.presort.param value is not a string" ,"960");
				}
				params['script.presort.param'] = req.body['script.presort.param'];
				delete req.body['script.presort.param'];
			}
		}
		else if (req.body.hasOwnProperty('script.presort.param')) {
			return errorHandler.handleError("BadRequest", req, res, "script.presort.param without script.presort" ,"960");		
		}

		var remainingProps = Object.keys(req.body);
		if (remainingProps.length > 0) {
			return errorHandler.handleError("BadRequest", req, res, "Unknown parameter(s): "+remainingProps.join(','),"960");		
		}
		
		try {
			util.thrift_client.createRecord(
				params,
				function(thrifError, thrifResult){
					return util.handleThrifReturn(thrifError, thrifResult, req, res);
				}
			);
		} catch (err) {
			util.thriftExceptionHandler(err, req, res);
		}
	}
};
